"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WWW_AUTHENTICATE_HEADER_NAME = exports.Kerberos = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _coreHttpServer = require("@kbn/core-http-server");
var _coreHttpRouterServerInternal = require("@kbn/core-http-router-server-internal");
/*
 *    Copyright 2020 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

const WWW_AUTHENTICATE_HEADER_NAME = exports.WWW_AUTHENTICATE_HEADER_NAME = 'WWW-Authenticate';
class Kerberos {
  constructor(props) {
    (0, _defineProperty2.default)(this, "checkAuth", async (request, response, toolkit) => {
      const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
      if (sessionCookie.credentials && sessionCookie.credentials.authHeaderValue) {
        const headers = {
          authorization: sessionCookie.credentials.authHeaderValue
        };
        try {
          const authInfoResponse = await this.searchGuardBackend.authinfo(headers);
        } catch (e) {
          return this.authenticateWithSPNEGO(request, response, toolkit, sessionCookie);
        }
        const rawRequest = (0, _coreHttpRouterServerInternal.ensureRawRequest)(request);
        (0, _lodash.assign)(rawRequest.headers, headers);
        return toolkit.next();
      }
      return this.authenticateWithSPNEGO(request, response, toolkit, sessionCookie);
    });
    this.type = 'kerberos';
    this.logger = props.logger;
    this.config = props.config;
    this.searchGuardBackend = props.searchGuardBackend;
    this.authDebugEnabled = this.config.get('searchguard.auth.debug');
    this.basePath = props.basePath ? props.basePath : "/";
    this.sessionStorageFactory = props.sessionStorageFactory;
  }

  // See the Negotiate Operation Example for the authentication flow details
  // https://tools.ietf.org/html/rfc4559#section-5

  async authenticateWithSPNEGO(request, response, toolkit, sessionCookie) {
    let backendError;
    try {
      const whitelistRoutes = this.config.get('searchguard.auth.unauthenticated_routes');
      if (whitelistRoutes.includes(request.route.path)) {
        return toolkit.next();
      }
      const headers = {};
      if (request.headers.authorization) {
        headers.authorization = request.headers.authorization;
      }

      // Validate the request.
      // The headers.authorization may hold SPNEGO GSSAPI token or basic auth credentials.
      const createSessionResponse = await this.searchGuardBackend.createSessionWithHeaders(headers);
      if (this.authDebugEnabled) {
        this.logger.debug(`Authenticated: ${JSON.stringify(createSessionResponse, null, 2)}.`);
      }
      const authHeaders = {
        authorization: 'Bearer ' + createSessionResponse.token
      };
      const user = await this.searchGuardBackend.authenticateWithHeader("authorization", authHeaders.authorization);
      sessionCookie.username = user.username;
      sessionCookie.credentials = {
        authHeaderValue: authHeaders.authorization
      };
      await this.sessionStorageFactory.asScoped(request).set(sessionCookie);
      const rawRequest = (0, _coreHttpRouterServerInternal.ensureRawRequest)(request);
      (0, _lodash.assign)(rawRequest.headers, headers);
      return toolkit.next();
    } catch (error) {
      backendError = error.inner || error;
      if (backendError && backendError.meta && backendError.meta.headers["www-authenticate"]) {
        let authenticateHeader = backendError.meta.headers["www-authenticate"];
        let parts = authenticateHeader.split(/\s*,\s*/);
        for (let negotiationProposal of parts) {
          if (negotiationProposal.startsWith('Negotiate')) {
            return response.unauthorized({
              headers: {
                [WWW_AUTHENTICATE_HEADER_NAME]: negotiationProposal
              }
            });
          }
        }
      }
      return response.unauthorized({
        body: backendError
      });
    }
  }
  async getCookieWithCredentials(request) {
    const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
    if (sessionCookie.credentials && sessionCookie.credentials.authHeaderValue) {
      const headers = {
        authorization: sessionCookie.credentials.authHeaderValue
      };
      try {
        const authInfoResponse = await this.searchGuardBackend.authinfo(headers);
      } catch (e) {
        return this.authenticateWithSPNEGOtoCookie(request, sessionCookie);
      }
      return sessionCookie;
    }
    return this.authenticateWithSPNEGOtoCookie(request, sessionCookie);
  }
  async authenticateWithSPNEGOtoCookie(request, sessionCookie) {
    try {
      const headers = {};
      if (request.headers.authorization) {
        headers.authorization = request.headers.authorization;
      }

      // Validate the request.
      // The headers.authorization may hold SPNEGO GSSAPI token or basic auth credentials.
      const createSessionResponse = await this.searchGuardBackend.createSessionWithHeaders(headers);
      if (this.authDebugEnabled) {
        this.logger.debug(`Authenticated: ${JSON.stringify(createSessionResponse, null, 2)}.`);
      }
      const authHeaders = {
        authorization: 'Bearer ' + createSessionResponse.token
      };
      const user = await this.searchGuardBackend.authenticateWithHeader("authorization", authHeaders.authorization);
      sessionCookie.username = user.username;
      sessionCookie.credentials = {
        authHeaderValue: authHeaders.authorization
      };
      return sessionCookie;
    } catch (error) {
      console.error(error);
      return sessionCookie;
    }
  }
  getAuthHeader(session) {
    if (session.credentials && session.credentials.authHeaderValue) {
      return {
        authorization: session.credentials.authHeaderValue
      };
    }
    return false;
  }
}
exports.Kerberos = Kerberos;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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